{$Define debug}
{$X+}
{$M 2048,0,0}
{$IFDEF Debug}
  {$A+,B-,D+,E-,F-,G+,I+,L+,N-,O-,P-,Q+,R+,S+,T-,V-,X+,Y+}
{$ELSE}
  {$A+,B-,D-,E-,F-,G+,I-,L-,N-,O-,P-,Q-,R-,S-,T-,V-,X+,Y-}
{$ENDIF}

(* Stand: 3.1.96 *)

program BBB;

{$IFDEF Debug}
Uses Crt, Dos;
{$ENDIF}

type
  TColor = ( cWin, cText, cOption, cLight, cHot);
  String8 = String[8];
  String16 = String[16];
  String31 = String[36];
  String128 = String[128];

type TColors = array[ TColor] of Byte;

const
  Version = '1.0';
  Colors: TColors =
    ( $9f, $8f, $f0, $f9, $0c);                            (* Default *)
{    ( $3f, $b0, $f0, $f3, $0d);}
  DarkColors: TColors =
    ( $1f, $4f, $70, $7f, $0e);                            (* non VGA *)
  MonoColors: TColors =
    ( $70, $07, $07, $07, $01);                               (* Mono *)

  Config: PChar = Ptr($8800, 0000);                   (* Novell DOS 7 *)
  TimeOut: Integer = 0;

const
  MaxVars = 128;
  MaxElems = MaxVars;
  Border=4;

var
  ScanLine: PChar;                                    (* Current Line *)
  LineNr: Word;                                       (* Current Line *)
  ScanOk: Boolean;                                    (* option found *)
  ErrorLine: PChar;
  Errors: Word;
  BBBLine: PChar;                                      (* call to bbb *)
  EndOfFile: PChar;

type
  TElemNr = -1..MaxElems;

var
  Elems: TElemNr;
  DlgHeight: Integer;
  DlgWidth: Integer;

var
  upper, key: array[char] of char;              (* translation tables *)
  CharHeight: Byte;
  UnderLine: Byte;
  OrgPal: Array[0..16] of Byte;                 (* mit OverScan *)
  NewPal: Array[0..16] of Byte;
  Used: Array[0..16] of Boolean;
  HotColors: TColors;

(* Low level ***************************************************************)

function IsVGA: Boolean; assembler;
asm
        mov     ax,1A00h
        int     10h
        cmp     al,1Ah
        mov     al,True
        je      @end
        mov     al,False
@end:
end;

procedure SetBlink( Enable: Boolean); assembler;
asm
        mov     ax,1003h
        mov     bl,enable
        int     10h
end;

procedure Set8x8; assembler;
asm
        mov     ax,1112h
        mov     bl,1
        int     10h
        mov     ax,1112h
        mov     bl,0
        int     10h
        mov     CharHeight,8
end;

procedure Set8x16; assembler;
asm
        mov     ax,1114h
        mov     bl,1
        int     10h
        mov     ax,1114h
        mov     bl,0
        int     10h
        mov     CharHeight,16
end;

procedure SetCharSet( b: byte); assembler;
asm
        mov     ax,1103h
        mov     bl,b
        int     10h
end;

procedure SetPalReg( reg, col: byte); assembler;
asm
        mov     ax,1000h
        mov     bl,reg
        mov     bh,col
        int     10h
end;

procedure GetPal(var pal); assembler;
asm
        mov     ax,1009h
        les     dx,pal
        int     10h
end;

procedure SetPal( var pal); assembler;
asm
        mov     ax,1002h
        les     dx,pal
        int     10h
end;

procedure MoveCursor( x, y: Byte); assembler;
asm
  mov ah,2
  mov bh,0
  mov dh,y
  mov dl,x
  int $10
end;

function GetKey: Word; assembler;
asm
  mov ah,0
  int 16h
end;

function KeyPressed: Boolean; assembler;
asm
  mov ah,1
  int 16h
  mov ax,False
  jz @1
  mov ax,True
@1:
end;

function max( a, b: Byte): Byte; assembler;
asm
  mov al,a
  cmp al,b
  jnb @1
  mov al,b
@1:
end;

function min( a, b: Word): Word; assembler;
asm
  mov ax,a
  cmp ax,b
  jb @1
  mov ax,b
@1:
end;

(* Screen ******************************************************************)

const
  MonoBase = $3B4;

type
  TScreenElem = record
    c: Char;
    a: Byte;
  end;
  PScreen = ^TScreen;
  TScreen = array[0..MaxInt-1] of TScreenElem;

var
  Screen, SaveScreen: PScreen;
  SaveCursor: Word;
  ScreenSize: Word absolute $40:$4C;
  ScreenW   : Word absolute $40:$4A;
  ScreenBase: Word absolute $40:$63;
  ScreenHVGA: Byte absolute $40:$84;
  ScreenH   : Word;

function GetCursor: Word; assembler;
asm
  mov ah,3
  mov bh,0
  int 10h
  mov ax,dx
end;

procedure SetCursor( w: Word); assembler;
asm
  mov dx,w
  mov ah,2
  mov bh,0
  int 10h
end;

procedure GetFontAccess;
const SeqRegs : array[1..4] of word = ( $0100, $0402, $0704, $0300 );
      GCRegs  : array[1..3] of word = ( $0204, $0005, $0406 );
var i : byte;
begin
  asm CLI end;
  for i := 1 to 4 do portw[$3C4]:= SeqRegs[i];
  for i := 1 to 3 do portw[$3CE]:= GCRegs[i];
  asm sti end;
end;

procedure ReleaseFontAccess;
const SeqRegs : array[1..4] of word = ( $0100, $0302, $0304, $0300 );
      GCRegs  : array[1..3] of word = ( $0004, $1005, $0E06 );
var i : byte;
begin
  asm CLI end;
  for i:= 1 to 4 do portw[$3C4]:= SeqRegs[i];
  for i:= 1 to 3 do portw[$3CE]:= GCRegs[i];
  asm sti end;
end;

function FindPal(col: Byte; var pal: Byte): Boolean;
var found: Boolean;
    p: Byte;
begin
  found:= False;
  for p:= 0 to 15 do
    if Used[p] and (NewPal[p]=col) then begin
      found:= True;
      break;
    end;
  if not found then
    for p:= 0 to 15 do
      if not Used[p] then break;
  pal:= p;
  FindPal:= found;
end;

procedure UsePal(pal, col: Byte);
begin
  Used[Pal]:= True;
  NewPal[Pal]:= col;
end;

procedure MakeNewPal;
var i: Integer;
    col, bg, fg: Byte;
    o: TColor;
begin
  FillChar( Used, SizeOf( used), 0);
  UsePal( 0, 0);
  UsePal( 7, 7);
  for o:= cWin to cLight do begin                   (* foreground *)
    col:= Colors[o] and $0F;
    if not FindPal(col, fg) then UsePal( fg, col);
  end;
  for o:= cWin to cLight do begin                   (* background *)
    col:= Colors[o] shr 4;
    if not FindPal(col, bg) then UsePal( bg, col);
  end;
  for o:= cWin to cLight do                         (* HotColors*)
    UsePal(15-Integer(o), HotColors[o] and $0F);
  for o:= cWin to cLight do begin                   (* map colors *)
    FindPal(Colors[o] and $0F, fg);
    FindPal(Colors[o] shr 4, bg);
    Colors[o]:= bg shl 4 + fg;
  end;
  for i:= 0 to 15 do                                (* map pal *)
    NewPal[i]:= OrgPal[NewPal[i]];
  for o:= cWin to cLight do
    HotColors[o]:= (Colors[o] and $f0) or (15-Integer(o));
  SetPal( NewPal);
end;

procedure DoUnderline;
var font, f: ^Byte;
    c: Char;
    i, j, b: Integer;
begin
  GetFontAccess;
  if CharHeight=16 then font:= ptr( $A000, $4000+14)
  else font:= ptr( $A000, $4000+7);
  for j:= 1 to UnderLine do begin
    f:= font;
    for i:= 0 to 255 do begin
      f^:= $FF;
      Inc( f, 32);
    end;
    Dec( font);
  end;
  ReleaseFontAccess;
  MakeNewPal;
end;


procedure InitScreen;
var c: TColor;
begin
  if isVGA then
    if (DlgHeight>23) then Set8x8
    else Set8x16;
  SaveScreen:= Pointer(PChar(Screen)+ScreenSize);
  Move( Screen^, SaveScreen^, ScreenSize);
  SaveCursor:= GetCursor;
  if isVGA and not (ScreenBase=MonoBase) then begin
    SetBlink( False);
    GetPal(OrgPal);
    if UnderLine>0 then begin
      DoUnderline;
      SetCharSet( 0 + 1*4 );
    end else
    for c:= cWin to cLight do
      if (HotColors[c] and $0f)=(Colors[c] and $0f) then
        HotColors[c]:= (Colors[c] and $f0) or (Colors[cHot] and $0f);
  end;
end;

procedure DoneScreen;
begin
  Move( SaveScreen^, Screen^, ScreenSize);
  if CharHeight=8 then Set8x16;
  SetCursor( SaveCursor);
  if IsVGA and not (ScreenBase=MonoBase) then begin
    SetBlink( True);
    SetCharSet(0);
    SetPal(OrgPal);
  end;
end;

procedure BiosPrint( c: char); assembler;
asm
    mov ah,0eh
    mov al,c
    cmp al,9
    jne @1
    mov al,' '
@1: int 10h
end;

procedure Print( const s: String);
var i: Integer;
begin
  for i:= 1 to length(s) do BiosPrint(s[i]);
end;

procedure PrintNr( i: Integer);
var nr: string8;
begin
  str( i, nr);
  Print( nr);
end;

procedure ClrLine;
var i: Integer;
begin
  BiosPrint( #13);
  for i:=2 to ScreenW do BiosPrint( ' ');
  BiosPrint( #13);
end;

procedure Wait;
begin
  Print( 'Press a key to continue');
  GetKey;
  ClrLine;
  BiosPrint( #10);
end;

procedure Error( const err: string);
begin
  Print( 'Error: ');
  Print( err);
  Print( #13#10);
  Inc( Errors);
  if Errors>10 then begin
    Wait;
    Errors:= 1;
  end;
end;

procedure Warning( const warn: string);
begin
  Print( 'Warning: ');
  Print( warn);
  Print( #13#10);
end;

procedure ScanError( const err: string);
var p: PChar;
    i: Integer;
begin
  if ScanLine<>ErrorLine then begin
    ErrorLine:= ScanLine;
    Print( 'Error in line ');
    PrintNr( LineNr);
    Print( ' ');
    p:= ErrorLine; i:= 50;
    while not (p^ in [#$0a,#$0d,#$1a]) and (i>0) do begin
      BiosPrint( p^);
      Inc(p);
      Dec(i);
    end;
    if i=0 then print( '...');
    Print( ''#13#10);
  end;
  Print( '  - ');
  Print( err);
  Print( #13#10);
  Inc( Errors);
  if Errors>10 then begin
    Wait;
    Errors:= 1;
  end;
end;

(* Output ******************************************************************)

var
  PosX, PosY: Word;                                    (* current Position *)
  Color: Byte;                                            (* current Color *)
  CurCol: TColor;

procedure SetColor( c: TColor);
begin
  if c=cHot then Color:= HotColors[CurCol]
  else Color:= Colors[c];
  CurCol:= c;
end;

procedure PutChar( ch: Char);
begin
  with Screen^[PosY*ScreenW+PosX] do begin
    c:= ch;
    a:= Color;
  end;
  Inc( PosX);
end;

procedure PutString( const s: String128; Hot: Byte);
var i: Integer;
    c: Byte;
begin
  if hot>0 then begin
    for i:= 1 to Hot-1 do PutChar( s[i]);
    c:= Color;
    SetColor( cHot);
    PutChar( s[Hot]);
    Color:= c;
  end;
  for i:= Hot+1 to Length( s) do PutChar( s[i]);
end;

procedure CenterString( const s: String128; Hot: Byte);
begin
  PosX:= (ScreenW-Length(s)) div 2+1;
  PutString( s, Hot);
  Inc( PosY);
end;

procedure HideCursor;
begin
  MoveCursor( 0, ScreenH+1);
end;

(* Window ******************************************************************)

type
  TLineChars = array[0..2] of Char;

var
  WinX, WinY, WinW, WinH, WinB: Word;

procedure Line( Chars: TLineChars);
var w: Word;
begin
  SetColor( cWin);
  PosX:= WinX-2; w:= WinW+2;
  PutChar( ' ');
  PutChar( Chars[0]);
  while w>2 do begin
    PutChar( Chars[1]);
    Dec( w);
  end;
  PutChar( Chars[2]);
  PutChar( ' ');
  PosX:= WinX; Inc( PosY);
end;

procedure CenterWin( w, h: Integer);
begin
  WinW:= w; WinH:= h;
  WinX:= (ScreenW-w) div 2+1;
  WinY:= (ScreenH+1-h) div 2;
  PosY:= WinY;
  Line( 'ͻ');
  while h>2 do begin
    Line( ' ');
    Dec( h);
  end;
  Line( 'ͼ');
end;

(* Variables ***************************************************************)

type
  TVarFlag = (vfExport, vfDisk);
  TVar = record
    Name: String8;
    Value: Byte;
    Flags: Set of TVarFlag;
  end;

type
  TVarNr = 0..MaxVars;
  TVarArray = array[0..MaxVars-1] of TVar;

var
  VarArray: TVarArray;
  Vars: TVarNr;
  VarFlags: Set of TVarFlag;

procedure InitVar( const aName: String8; aValue: Byte);
var i: Integer;
begin
  with VarArray[Vars] do begin
    Name:= aName;
    Value:= aValue;
    Flags:= [];
  end;
end;

function FindVar( aName: String8; var Nr: TVarNr): Boolean;
var i: Integer;
    len: Byte absolute aName;
begin
  FindVar:= True;
  while (len>0) and (aName[len]=' ') do dec(len);
  for i:= 1 to length(aName) do aName[i]:= Upper[aName[i]];
  for i:= 0 to Vars-1 do
    if VarArray[i].Name=aName then begin
      Nr:= i;
      exit;
    end;
  FindVar:= False;
end;

(* Quickies ****************************************************************)

const
  MaxQuicks = 128;

type
  TQuick = record
    VarNr: TVarNr;
    Value: Byte;
  end;
  TQuickNr = 0..MaxQuicks;
  TQuickArray = array[0..MaxQuicks-1] of TQuick;

var
  QuickArray: TQuickArray;
  Quicks: TQuickNr;

function IsQuickSet( first, last: TQuickNr): Boolean;
var b: Boolean;
    q: TQuickNr;
begin
  b:= True;
  for q:= first to last do with QuickArray[q] do
    b:= b and (VarArray[VarNr].Value=Value);
  IsQuickSet:= b;
end;

(* Dialog Elements *********************************************************)

type
  TElemTyp = (elCheckBox, elRadioButton, elQuick, elSpace, elText);

  TElem = object
    Typ: TElemTyp;
    Title: String31;
    HotPos: Byte;
    HotKey: Char;
    Para1, Para2: Byte;
    procedure SetTitle( aTitle: String31);
    procedure Init( aTyp: TElemTyp; aPara1, aPara2: Byte);
    procedure Paint;
    procedure Press;
  end;

const
  activeElems: Set of TElemTyp = [elCheckBox, elRadioButton, elQuick];

type
  TElemArray  = array[0..MaxElems-1] of TElem;

var
  ElemArray: TElemArray;
  Current: TElemNr;

procedure TElem.Init( aTyp: TElemTyp; aPara1, aPara2: Byte);
begin
  Typ:= aTyp;
  Para1:= aPara1;
  Para2:= aPara2;
end;

procedure TElem.SetTitle( aTitle: String31);
begin
  while (aTitle[1]=' ') and (length(aTitle)>0) do Delete(aTitle, 1, 1);
  if aTitle[1]='!' then begin
    Delete(aTitle, 1, 1);
    Press;
  end;
  while (aTitle[1]=' ') and (length(aTitle)>0) do Delete(aTitle, 1, 1);
  case Typ of
    elCheckBox: Title:= ' [ ] '+aTitle;
    elRadioButton: Title:= ' ( ) '+aTitle;
    elQuick: Title:= ' < > '+aTitle;
    else Title:= aTitle;
  end;
  HotPos:= Pos( '_', Title);
  if HotPos>0 then begin
    Delete( Title, HotPos, 1);
    HotKey:= Upper[Title[ HotPos]];
  end else
    HotKey:= #0;
end;

procedure TElem.Paint;
const Colors: Array[TElemTyp] of TColor = (cOption, cOption, cOption, cWin, cText);
var selected: Boolean;
begin
  selected:= @self=@ElemArray[Current];
  if selected then begin
    SetColor(cLight);
    MoveCursor( PosX+2, PosY);
  end else SetColor( Colors[Typ]);
  case Typ of
    elRadioButton: if Para2=VarArray[Para1].Value
      then Title[3]:= ''
      else Title[3]:= ' ';
    elCheckBox: if VarArray[Para1].Value=0
      then Title[3]:= ' '
      else Title[3]:= 'X';
    elQuick: if IsQuickSet(Para1, Para2)
      then Title[3]:= ''
      else Title[3]:= ' ';
  end;
  PutString( Title, HotPos);
end;

procedure TElem.Press;
var q: TQuickNr;
begin
  case Typ of
    elRadioButton:
      VarArray[Para1].Value:= Para2;
    elCheckBox:
      if VarArray[Para1].Value>0
        then VarArray[Para1].Value:= 0
        else VarArray[Para1].Value:= 1;
    elQuick: for q:= Para1 to Para2 do with QuickArray[q] do
      VarArray[VarNr].Value:=Value;
  end;
end;

(* Dialog ******************************************************************)

const
  MaxColumns = 8;

type
  TColumn = record
    FirstElem: TElemNr;
    LastElem: TElemNr;
    Height: Integer;
    Width: Integer;
  end;
  TColumnNr = 0..MaxColumns;

var
  ColumnArray: array[0..MaxColumns-1] of TColumn;
  Columns: TColumnNr;

procedure AddColumn;
var elem: TElemNr;
    found: Boolean;
begin
  if Columns<MaxColumns then with ColumnArray[Columns] do begin
    if Columns=0 then firstElem:= 0
      else firstElem:= ColumnArray[Columns-1].LastElem;
    found:= False;
    Width:= 0;
    for elem:= firstElem to Elems-1 do with ElemArray[elem] do begin
      if Typ in activeElems then found:= True;
      Width:= max( Width, length( Title));
    end;
    if found then begin
      LastElem:= Elems;
      Height:= LastElem-FirstElem;
      DlgHeight:= max(DlgHeight, Height);
    end;
    if not found or (DlgWidth+Width+Border>=ScreenW-4-Border) then begin
      if not found then Error( 'no selectable element in column')
      else Error( 'dialog too wide, cannot create column');
      if Columns=0 then exit;
      Dec(Columns);
      Dec( DlgWidth, ColumnArray[Columns].Width+Border);
      AddColumn;
      exit;
    end;
    Inc( DlgWidth, Width+Border);
    Inc(Columns);
  end else Error( 'too many columns')
end;

procedure PaintElems;
var i, j, x, e: Integer;
begin
  x:= WinX+WinB; e:= 0;
  for j:= 0 to Columns-1 do with ColumnArray[j] do begin
    PosY:= WinY+1;
    for i:= 0 to Height-1 do begin
      PosX:= x;
      ElemArray[e].Paint;
      while PosX<x+Width do PutChar(' ');
      Inc(PosY);
      Inc(e);
    end;
    inc( x, Width+Border);
  end;
end;

procedure PaintDialog;
var wWidth, i: Integer;
begin
  wWidth:= max( DlgWidth+Border, 31);
  WinB:= (wWidth-DlgWidth) div 2;
  CenterWin( wWidth, DlgHeight+2);
  PosY:= WinY;
  CenterString(' BBB v'+Version+' ', 0);
  PosY:= WinY+WinH-1;
  CenterString(' (c) 1995 Bert Schnwlder ', 0);
  PaintElems;
end;

(***************************************************************************)

var
  CurrentColumn: TColumnNr;

procedure Select( elem: TElemNr; delta: Integer);
begin
  Current:= (elem+Elems) mod Elems;
  while not (ElemArray[Current].Typ in activeElems) do (* search for option *)
    Current:= (Current+Elems+delta) mod Elems;
  for CurrentColumn:= 0 to Columns do with ColumnArray[CurrentColumn] do
    if Current<LastElem then break;
  PaintElems;
end;

procedure SelectColumn( delta: Integer);
var elem: TElemNr;
    c: TColumnNr;
begin
  elem:= Current-ColumnArray[CurrentColumn].FirstElem;
  CurrentColumn:= (CurrentColumn+Columns+delta) mod Columns;
  with ColumnArray[CurrentColumn] do
    if elem>=Height-1 then Select(LastElem-1, -1)
    else begin
      c:= CurrentColumn;
      Select(FirstElem+Elem, delta);
      if c<>CurrentColumn then Select(FirstElem+Elem, -delta);
    end;
end;

procedure PressCurrent;
begin
  ElemArray[Current].Press;
  PaintElems;
end;

procedure NextGroup( delta: Integer);
var elem: TElemNr;
begin
  elem:= Current;
  repeat
    Inc( elem, delta);
  until (elem<0) or (elem=Elems) or (elem=Current)
    or not (ElemArray[elem].Typ in activeElems);
  if elem<0 then elem:= 0;
  if elem=Elems then dec(elem);
  repeat
    elem:= (elem+Elems+delta) mod Elems;
  until (ElemArray[elem].Typ in activeElems) or (elem=Current);
  if elem<>Current then Select(elem, delta);
end;

procedure MainLoop;
var c: Word;
    done: Boolean;
    i: Integer;
begin
  Select( 0, +1);
  done:= False;
  repeat
    c:= GetKey;
    case char(c) of
      #13, #27: done:= True;
      ' ': PressCurrent;
      #9: NextGroup(+1);
      #0: case char(hi(c)) of
        #72: Select( Current+Elems-1, -1);
        #80: Select( Current+1, +1);
        #75: SelectColumn( -1);
        #77: SelectColumn( +1);
        #71: Select( 0, +1);
        #79: Select( Elems-1, -1);
        #15: NextGroup(-1);
      end;
    else
      for i:=0 to Elems-1 do with ElemArray[i] do
        if HotKey=Upper[key[char(c)]] then begin
          Press;
          Select( i, +1);
          break;
        end;
    end;
  until done;
end;

procedure TimerLoop;
var t: Byte;
    i: Integer;
    Timer: Byte absolute $40:$6c;
begin
  Select( 0, +1);
  TimeOut:= (TimeOut*183) div 10;
  t:= Timer;
  SetColor( cWin);
  PosY:= WinY;
  repeat
    if t<>Timer then begin
      t:= Timer;
      Dec( TimeOut);
      if TimeOut=18 then exit;
      PosX:= WinX+WinW-2;
      i:= (TimeOut * 10) div 183;
      repeat
        PutChar(char(i mod 10+ord('0')));
        Dec(PosX,2);
        i:= i div 10;
      until i=0;
      PutChar(' ');
    end;
  until keyPressed;
  PosX:= WinX+WinW-4;
  PutString( '', 0);
end;

(***************************************************************************)

var
  LineHead: String16;
  LineTail: String128;
  LineDev: Char;

procedure SplitLine( const s: String128);
var p, i: Integer;
begin
  i:= 1;
  while (i<=length(s)) and (s[i]=' ') do Inc(i);             (* skip space *)
  p:= i;
  LineHead:= ''; LineDev:= ' ';
  while (i<=length(s)) and not (s[i]in [' ','=']) do Inc(i);   (* get word *)
  if p=i then exit;
  LineHead:= Copy(s, p, i-p);
  while (i<=length(s)) and (s[i] in [' ','=']) do begin      (* skip space *)
    if s[i]='=' then LineDev:= '=';
    Inc(i);
  end;
  LineTail:= Copy( s, i, 255);
  while LineTail[length(LineTail)]=' ' do Dec( LineTail[0]);       (* trim *)
end;

procedure AddCheckBox( const varName: String8; const aTitle: String31);
begin
  if (Vars<MaxVars) and (Elems<MaxElems) then begin
    InitVar( varName, 0);
    with ElemArray[Elems] do begin
      Init( elCheckBox, Vars, 0);
      SetTitle( aTitle);
    end;
    Inc( Elems);
    Inc( Vars);
  end;
end;

procedure AddRadioButton( const varName: String8; const Titles: String128);
var button, i, p: Integer;
begin
  if (Vars<MaxVars) and (Elems<MaxElems) then begin
    InitVar( varName, 1);
    i:= 1;
    button:= 1;
    while (i<length(Titles)) and (button<10) and (Elems<MaxElems) do begin
      p:= i;
      while (Titles[i]<>'|') and (i<=length(Titles)) do Inc(i);
      with ElemArray[Elems] do begin
        Init( elRadioButton, Vars, button);
        SetTitle( Copy( Titles, p, i-p));
      end;
      Inc( i);
      Inc( button);
      Inc( Elems);
    end;
    Inc( Vars);
  end;
end;

procedure AddOption( const s: String128);
var vs: String8;
    os: String128;
    p, i: Integer;
begin
  p:= Pos(':', s);
  if p<1 then begin ScanError( '":" expected'); exit; end;
  vs:= Copy(s, 1, p-1);
  while vs[length(vs)]=' ' do Dec(vs[0]);
  for i:= 1 to length(vs) do begin
    vs[i]:= UpCase(vs[i]);
    if not (vs[i] in ['A'..'Z', '_']) then begin
      ScanError( '"'+vs+'" is not a valid identifier');
      exit;
    end;
  end;
  os:= Copy(s, p+1, 255);
  if Pos('|', os)>0
    then AddRadioButton( vs, os)
    else AddCheckBox( vs, os);
  ScanOk:= True;
end;

procedure AddQuick( const s: String128);
var p, i: Integer;
    first: TQuickNr;
    aTitle: String31;
    Nr: TVarNr;
begin
  if (Elems<MaxElems) then begin
    p:= Pos(':', s);
    if p<1 then begin ScanError( '":" expected'); exit; end;
    aTitle:= Copy(s, p+1, 255);
    first:= Quicks;
    SplitLine( Copy(s, 1, p-1));
    while LineHead<>'' do begin
      if not FindVar(LineHead, nr) then
        ScanError('unknown variable: '+LineHead)
      else begin
        if LineDev<>'=' then
          ScanError( '"=" expected')
        else begin
          if LineTail[1] in ['0'..'9'] then begin
            QuickArray[Quicks].VarNr:= nr;
            QuickArray[Quicks].Value:= Ord(LineTail[1])-Ord('0');
            Inc( Quicks);
          end
          else ScanError( 'not a valid value: '+ LineTail[1]);
        end;
      end;
      Delete( LineTail, 1, 1);
      SplitLine( LineTail);
    end;
    if Quicks>first then with ElemArray[Elems] do begin
      Init( elQuick, first, Quicks-1);
      SetTitle( aTitle);
      Inc( Elems);
    end;
  end;
end;


procedure AddFormat( aTyp: TElemTyp; const s: String31);
begin
  if Elems<MaxElems then begin
    with ElemArray[Elems] do begin
      Init( aTyp, 0, 0);
      SetTitle( s);
    end;
    Inc( Elems);
  end;
end;

(***************************************************************************)
{$IFDEF Debug}
var
  buf: Array[0..3000] of byte;

procedure LoadConfig;
var f: file;
    read: Word;
begin
  assign(f, 'config.bbb');
  reset(f,1);
  blockread( f, buf, sizeof(buf), read);
  close(f);
  Config:= @buf;
  buf[read]:= $1A;
end;

procedure view( p: PChar);
var opt: String31;
begin
  Str(LongInt(p), opt);
  SwapVectors;
  Exec( 'BBBVIEW.EXE', opt);
  SwapVectors;
end;
{$ENDIF}

(* Conditions **************************************************************)

const
  MaxConds = MaxElems;

type
  TCondNr = -1..MaxConds;

var
  CondArray: array[0..MaxConds-1] of Boolean;
  CondLine: array[0..MaxConds-1] of PChar;
  CondLineNr: array[0..MaxConds-1] of Word;
  Conds: TCondNr;
  CondStatus: Boolean;

function Evaluate( cond: String128): Boolean;
var p: Integer;
    name: String8;
    nr: TVarNr;
    len: Byte absolute cond;
    invert: Boolean;
begin
  Evaluate:= False;
  if Pos('=', cond)=0 then
    if cond[1]='!' then cond:= copy( cond, 2, 255)+'==0'
    else cond:= cond+'!=0';
  repeat
    p:= Pos(' ', cond);
    if p>0 then delete( cond, p, 1);
  until p=0;
  p:= Pos('=', cond);
  name:= Copy(cond, 1, p-1);
  invert:= cond[p-1] = '!';
  Inc(p);
  if invert then dec(name[0])
  else if cond[p]='=' then Inc(p);
  if FindVar( name, nr) then begin
    if (p<=len) and (cond[p] in ['0'..'9']) then
      Evaluate:= (VarArray[nr].Value+Ord('0') = Ord(cond[p])) xor invert
     else ScanError('cannot evaluate condition: '+cond);
  end else ScanError('unknown variable: '+name);
end;

procedure CalcStatus;
var i: TCondNr;
begin
  CondStatus:= True;
  for i:= 0 to Conds-1 do
    CondStatus:= CondStatus AND CondArray[i];
end;

procedure InsertIf( var cond: String128);
begin
  if Conds<MaxConds then begin
    CondArray[ Conds]:= Evaluate( cond);
    CondLine[ Conds]:= ScanLine;
    CondLineNr[ Conds]:= LineNr;
    Inc( Conds);
    CalcStatus;
  end;
end;

procedure EndIf;
begin
  if Conds>0 then begin
    Dec( Conds);
    CalcStatus;
  end else
    ScanError( 'ENDIF without IF')
end;

procedure InvertIf;
begin
  if Conds>0 then begin
    CondArray[Conds-1]:= not CondArray[Conds-1];
    CondLine[Conds-1]:= ScanLine;
    CondLineNr[ Conds-1]:= LineNr;
    CalcStatus;
  end else
    ScanError( 'ELSE without IF')
end;

procedure ElseIf( var cond: String128);
begin
  if (Conds>0) then begin
    if CondArray[Conds-1]
      then CondArray[Conds-1]:= False
      else CondArray[Conds-1]:= Evaluate( cond);
    CondLine[Conds-1]:= ScanLine;
    CondLineNr[ Conds-1]:= LineNr;
    CalcStatus;
  end else
    ScanError( 'ELSEIF without IF')
end;

(***************************************************************************)

procedure PrintChr( var dest: PChar; source: Char);
begin
  dest^:= source;
  Inc( dest);
end;

procedure PrintStr( var dest: PChar; const source: String128);
var i: Integer;
begin
  for i:= 1 to length( source) do
    PrintChr( dest, source[i]);
end;

procedure ExportVars;
var i: Integer;
begin
  if not (vfExport in VarFlags) then Exit;
  for i:= 0 to Vars-1 do with VarArray[i] do
    if vfExport in Flags then begin
      PrintStr( BBBLine, 'SET ');
      PrintStr( BBBLine, Name);
      PrintChr( BBBLine, '=');
      PrintChr( BBBLine, Char(Value+Ord('0')));
      PrintStr( BBBLine, #13#10);
    end;
end;

procedure FillToEof;
const bits: Array[0..7] of LongInt = (
        $33333200, $33001320, $33002310, $33333200,
        $33001320, $33002310, $33333100, $22222222);
      chars: Array[0..3] of Char= (' ', '', '', '');
var i, j, y: Integer;
    m: LongInt;
begin
  if BBBLine<=EndOfFile then begin
    PrintStr( BBBLine, #13#10'REM --- now fill to the end of file ---'#13#10);
    repeat
      PrintStr( BBBLine, #13#10);
      for y:= 0 to 7 do begin
        PrintStr( BBBLine, 'REM ');
        for i:= 0 to 5 do begin
          if i=3 then for j:= 1 to 7 do PrintChr( BBBLine, ' ');
          m:= bits[y];
          for j:= 0 to 7 do begin
            PrintChr( BBBLine, chars[(m shr 28) and 3]);
            m:= m shl 4;
          end;
        end;
        PrintStr( BBBLine, #13#10);
      end;
    until BBBLine>=EndOfFile;
  end;
  PrintChr(BBBLine, #$1a);
end;

(***************************************************************************)

var buffer: array[0..$8000] of char;
    bufsize: Word;

procedure CreateCopy;
var source, dest: PChar;
    c: Char;
begin
  while not (BBBLine^ in [#$0a, #$0d, #$1A]) do Inc(BBBLine);
  while BBBLine^ in [#$0a, #$0d] do Inc(BBBLine);
  source:= BBBLine;
  dest:= Buffer;
  bufSize:= 0;
  while (source^<>#$1a) and (BufSize<SizeOf(buffer)-1) do begin
    dest^:= source^;
    Inc( dest); Inc( source); Inc(BufSize);
  end;
  dest^:= #$1a;
end;

(***************************************************************************)

const
  StackSize = 64;

var
  Stack: array[0..StackSize] of PChar;
  GosubLine: array[0..StackSize] of PChar;
  GosubLineNr: array[0..StackSize] of Word;
  SP: Word;
  SearchLabel: String8;                                  (* when Jumping *)
  SearchFrom: PChar;                                       (* line start *)
  SearchFromNr: Word;
  SearchIP: PChar;

function InitSearch( const para: String8; IP: PChar): Boolean;
var i: Integer;
begin
  InitSearch:= False;
  if para='' then ScanError('no label')
  else begin
    SearchLabel:= para;
    for i:= 1 to length(SearchLabel) do
      SearchLabel[i]:= UpCase(para[i]);
    SearchFrom:= ScanLine;
    SearchIP:= IP;
    InitSearch:= True;
  end;
end;

procedure StartGosub( const para: String8; IP: PChar);
begin
  if InitSearch(para, IP) then begin
    Stack[SP]:= IP;
    GosubLine[SP]:= ScanLine;
    GosubLineNr[SP]:= LineNr;
    Inc( SP);
  end;
end;

procedure EndGosub( var IP: PChar);
begin
  if SP>0 then begin
    Dec( SP);
    IP:= Stack[SP];
  end else ScanError( 'RETURN without GOSUB');
end;

(* Scanning - Pass 1 creates dialog, Pass 2 executes, Pass 3 flattens it ***)

procedure MarkVars( vf: TVarFlag);
var Nr: TVarNr;
begin
  repeat
    SplitLine(LineTail);
    if (LineHead<>'') then
      if FindVar(LineHead, Nr) then begin
        Include( VarArray[Nr].Flags, vf);
        Include( VarFlags, vf);
      end else ScanError('unknown variable: '+LineHead);
  until LineTail='';
end;

procedure GetUpper;
begin
  repeat
    SplitLine(LineTail);
    if length(LineHead)>1 then
      Upper[ LineHead[1]] := LineHead[2];
  until LineTail='';
end;

procedure GetKeys;
begin
  repeat
    SplitLine(LineTail);
    if length(LineHead)>1 then
      key[ LineHead[1]] := LineHead[2];
  until LineTail='';
end;

procedure GetColor;
var c: TColor;
    i, p, digits: Integer;
    dig: array[1..3] of Byte;       (* bg, fg, hot *)
const hex: string[16] = '0123456789ABCDEF';
begin
  for c:= cWin to cHot do begin
    SplitLine(LineTail);
    if length(LineHead)>0 then begin
      digits:= min(length(LineHead), 3);
      for i:= 1 to digits do begin
        p:= Pos(Upcase(LineHead[i]), hex);
        if p>0 then dig[i]:= p-1 else
        begin ScanError('bad hex digit: '+LineHead[i]); exit; end;
      end;
      if Digits=1 then begin dig[2]:= dig[1]; dig[1]:= 0; end;
      if Digits<3 then dig[3]:= dig[2];
      Colors[c]:= dig[1] shl 4 or dig[2];
      HotColors[c]:= dig[1] shl 4 or dig[3];
    end;
    if LineTail='' then exit;
  end;
end;

procedure ReplaceOptions( var s: String128);
var i, p, r: Integer;
    len: Byte absolute s;
    c: String31;
    dest: PChar;
    l: Byte;
    err: Boolean;
begin
  l:= Len;
  err:= False;
  p:= Pos('{', s);
  while p>0 do begin
    r:= p;
    while (s[r]<>':') and (r<len) do inc(r);
    if s[r]<>':' then begin ScanError( '":" expected'); err:= TRUE; break end;
    c:= Copy(s, p+1, r-p-1);
    Delete( s, p, r-p+1);
    r:= p;
    while (s[r]<>'}') and (r<=len) do inc(r);
    if Evaluate(c) then p:=r;
    if s[r]<>'}' then begin ScanError( '"}" expected'); err:= TRUE; break end;
    Delete( s, p, r-p+1);
    p:= Pos('{', s);
  end;
  if err then len:= p-1;
  while len<l do s:= s+' ';
  dest:= ScanLine;
  PrintStr( dest, s);
end;

procedure ReplaceLine( const s: String128);
var i, p, r: Integer;
    len: Byte absolute s;
    c: String31;
    dest: PChar;
begin
  dest:= ScanLine;
  p:= Pos('"', s);
  if p>0 then begin
    repeat inc( p) until (s[p]<>' ') or (p=len);
    r:= p;
    while (s[r]<>'"') and (r<len) do inc(r);
    if (r=p) or (r=len) then begin ScanError('condition expected'); exit end;
    c:= Copy(s, p, r-p);
    if Evaluate(c)
      then for i:= 1 to r do printChr( dest, ' ')
      else printChr( dest, ';');
  end else
    ScanError('condition expected');
end;

procedure AddLine;
begin
  if LineHead='' then exit;
  PrintStr( BBBLine, LineHead);
  if LineTail<>'' then begin
    PrintChr( BBBLine, LineDev);
    PrintStr( BBBLine, LineTail);
  end;
  PrintStr( BBBLine, #13#10);
end;

type
  tKeyWord = (kwUnknown, kwOption, kwSpace, kwText, kwColumn, kwQuick,
    kwExport, kwDisk, kwKey, kwUpper, kwTimeOut, kwColor, kwLine, kwIf,
    kwElse, kwElseIf, kwEndIf, kwComment);

function GetKeyWord: tKeyWord;
var i: Integer;
begin
  for i:= 1 to length(LineHead) do LineHead[i]:= UpCase(LineHead[i]);
  if LineHead='OPTION'    then GetKeyWord:= kwOption else
  if LineHead='SPACE'     then GetKeyWord:= kwSpace else
  if LineHead='TEXT'      then GetKeyWord:= kwText else
  if LineHead='COLUMN'    then GetKeyWord:= kwColumn else
  if LineHead='QUICK'     then GetKeyWord:= kwQuick else
  if LineHead='EXPORT'    then GetKeyWord:= kwExport else
  if LineHead='DISK'      then GetKeyWord:= kwDisk else
  if LineHead='KEY'       then GetKeyWord:= kwKey else
  if LineHead='UPPER'     then GetKeyWord:= kwUpper else
  if LineHead='TIMEOUT'   then GetKeyWord:= kwTimeOut else
  if LineHead='COLOR'     then GetKeyWord:= kwColor else
  if LineHead='UNDERLINE' then GetKeyWord:= kwLine else
  if LineHead='IF'        then GetKeyWord:= kwIf else
  if LineHead='ELSE'      then GetKeyWord:= kwElse else
  if LineHead='ELSEIF'    then GetKeyWord:= kwElseIf else
  if LineHead='ENDIF'     then GetKeyWord:= kwEndIf else
  if LineHead[1] in ['A'..'Z'] then GetKeyWord:= kwUnknown else
  GetKeyWord:= kwComment;
end;


procedure Scan( Pass: Integer);
var p, MaxP: PChar;
    i: Integer;
    s: String128;
    key: TKeyWord;
    done: boolean;
begin
  if Pass<3 then p:= Config else p:= Buffer;
  MaxP:= p+$8000;
  LineNr:= 1;
  if Pass<3 then Errors:= 0;
  SearchLabel:= '';
  SP:= 0;
  done:= p^=#$1a;
  while not done do begin
    ScanLine:= p;
    i:= 1;
    while not (p^ in [#$0a, #$0d, #$1A]) and (i<128) do begin
      if p^<>#9 then s[i]:= p^ else s[i]:= ' ';
      Inc(p); Inc(i);
    end;
    s[0]:= char(i-1);
    SplitLine( s);
    if LineHead[1]=';' then begin
      Delete( LineHead, 1, 1);
      key:= GetKeyWord;
      if Pass=1 then
        case key of
          kwUnknown: ScanError('unknown keyword');
          kwOPTION: AddOption( LineTail);
          kwQUICK: AddQuick( LineTail);
          kwSPACE: AddFormat( elSpace, LineTail);
          kwTEXT: AddFormat( elText, LineTail);
          kwCOLUMN: AddColumn;
          kwEXPORT: MarkVars( vfExport);
          kwDISK: MarkVars( vfDisk);
          kwKEY: GetKeys;
          kwUPPER: GetUpper;
          kwTIMEOUT: val( copy(LineTail,1,3), TimeOut, i);
          kwCOLOR: GetColor;
          kwLINE: if LineTail[1] in ['0'..'9'] then
            UnderLine:= Ord(LineTail[1])-Ord('0')
          else ScanError('number expected');
        end
      else if Pass=2 then
        case key of
          kwIF: InsertIf( LineTail);
          kwELSE: InvertIf;
          kwELSEIF: ElseIf(LineTail);
          kwENDIF: EndIf;
        end;
    end else begin (* no ';' *)
      if (Pass=1) and (Pos('INSTALL', LineHead)>0) then begin
        for i:= 1 to length(LineTail) do begin
          LineTail[i]:= UpCase(LineTail[i]);
          if LineTail[i]=' ' then LineTail[0]:= char(i);  (* only get path *)
        end;
        if copy(LineTail, length(LineTail)-6, 7)='BBB.SYS' then
          if BBBLine=nil then BBBLine:= ScanLine
          else Warning('you shouldn''t load BBB twice');
      end else
      if (Pass=2) and not (ScanLine^ in [#$0a, #$0d, #$1A]) then begin
        if (conds>0) and (CondStatus=False) then ScanLine^ := ';'
        else begin
          if LineHead[1]='?' then ReplaceLine(s);
          if Pos('{', s)>0 then ReplaceOptions(s);
        end;
      end else
      if Pass=3 then begin
        for i:= 1 to length(LineHead) do LineHead[i]:= UpCase(LineHead[i]);
        if SearchLabel='' then begin                    (* not searching *)
          if LineHead='EXIT' then p:= MaxP else
          if LineHead='GOTO' then InitSearch(LineTail, p) else
          if LineHead='GOSUB' then StartGosub(LineTail, p) else
          if LineHead='RETURN' then EndGosub(p) else
          if (LineHead<>'REM') and (LineHead<>'TIMEOUT') and (LineHead[1]<>':')
            then AddLine;
        end else begin                                      (* searching *)
          if (LineHead[1]=':') and (Copy(LineHead,2,8)=SearchLabel)
            then SearchLabel:= '';                              (* found *)
        end;
      end;
    end;
    while p^ in [#$0a, #$0d] do begin if p^=#$0a then Inc(LineNr); Inc(p); end;
    done:= (p^=#$1A) or (p>=MaxP);
    if done and (SearchLabel<>'') then begin
      ScanLine:= SearchFrom;
      LineNr:= SearchFromNr;
      ScanError( 'label not found after this line');
      if (SP>0) and (GosubLine[SP-1]=SearchFrom) then Dec(SP);
      SearchLabel:= '';
      p:= SearchIP;
      done:= (p^=#$1A) or (p>=MaxP);
    end;
  end;
  if Pass=1 then EndOfFile:= p else
  if Pass=2 then for i:= 1 to Conds do begin
    ScanLine:= CondLine[i-1];
    LineNr:= CondLineNr[i-1];
    ScanError( 'missing ENDIF');
  end else
  if Pass=3 then for i:= 1 to SP do begin
    ScanLine:= GosubLine[i-1];
    LineNr:= GosubLineNr[i-1];
    ScanError( 'GOSUB without RETURN');
  end;
end;

(***************************************************************************)

const VarFile = 'BBB.CFG';

procedure SaveVars;
var f: Text;
    i: Integer;
begin
  if not (vfDisk in VarFlags) then exit;
  Assign( f, VarFile);
  Rewrite( f);
  if IOResult<>0 then exit;
  for i:= 0 to Vars-1 do with VarArray[i] do
    if vfDisk in Flags then
      WriteLn(f, Name, ' = ', Value);
  Close(f);
end;

procedure LoadVars;
var f: Text;
    nr: TVarNr;
    s: String31;
begin
  if not (vfDisk in VarFlags) then exit;
  Assign( f, VarFile);
  Reset( f);
  if IOResult<>0 then exit;
  while not eof(f) do begin
    ReadLn( f, s);
    SplitLine( s);
    if (LineHead<>'') and FindVar(LineHead, nr)
      and (LineTail[1] in ['0'..'9']) then
        VarArray[nr].Value := Ord(LineTail[1])-Ord('0');
  end;
  Close(f);
end;

(***************************************************************************)

procedure Init;
var c: Char;
    o: TColor;
begin
  Vars:= 0;
  VarFlags:= [];
  Elems:= 0;
  Conds:= 0;
  Quicks:= 0;
  Columns:= 0;
  DlgWidth:= -Border;
  DlgHeight:= 0;
  ScanOk:= False;
  ScanLine:= Config;
  LineNr:= 1;
  Errors:= 0;
  ErrorLine:= NIL;
  BBBLine:= NIL;
  for c:= #0 to #255 do begin
    Upper[c]:= UpCase(c);
    key[c]:= c;
  end;
  UnderLine:= 1;
  if ScreenBase=MonoBase then begin
    Screen:= Ptr( $B000, 0000);
    Colors:= MonoColors;
  end else begin
    Screen:= Ptr( $B800, 0000);
    if not isVGA then Colors:= DarkColors;
  end;
  if IsVGA
    then ScreenH:= ScreenHVGA
    else ScreenH:= ScreenSize div ScreenW div 2;
  for o:= cWin to cLight do
    HotColors[o]:= (Colors[o] and $f0) or Colors[cHot];
end;

(* Main ********************************************************************)

begin
  ClrScr;
  Print( #13#10'BBB v'+Version+' (c) 1995 Bert Schnwlder'#13#10#10);
{$IFDEF Debug}
  LoadConfig;
{$ENDIF}
  Init;
  Scan( 1);
  if BBBLine=nil then begin
    Error( 'Couldn''t find INSTALL=BBB.SYS - aborting...');
    Wait; exit;
  end;
  if ScanOk then begin
    AddColumn;
    LoadVars;
    if Errors>0 then Wait;
    InitScreen;
    PaintDialog;
    if TimeOut>0 then TimerLoop;
    if (TimeOut=0) OR KeyPressed then MainLoop;
    DoneScreen;
    Scan(2);                                                    (* execute *)
    CreateCopy;
    PrintStr( BBBLine, 'REM --- this was created by BBB ---'#13#10);
    ExportVars;
    Scan(3);                                            (* build up linear *)
    PrintStr( BBBLine, 'REM --- the end ---'#13#10);
    FillToEOF;
    SaveVars;
    if Errors>0 then Wait;
{$IFDEF Debug}
    View( Config);
{$ENDIF}
  end else begin
    Error('Error: No Option found');
    Wait;
  end;
end.
