"      NAME     Syntax-Highlighting
       AUTHOR   bert@isg.cs.uni-magdeburg.de
       URL     	http://www.cs.uni-magdeburg.de/~bert/smalltalk.html
       FUNCTION highlights methods in all browsers
       ST-VERSIONS      VisualWorks 2.5 (maybe earlier, not tested yet)
       PREREQUISITES    (none)
       CONFLICTS        (none known)
       DISTRIBUTION     world
       VERSION  1.0 (for archive, internal  3.2 :)
       DATE     15-Nov-96

SUMMARY

Replaces the Text>makeSelectorBoldIn:
(the method which causes the selector of
a method to appear bold in a browser) with one
that does excessive (though efficient, I think)
syntax highlighting. It's much more easy to use
than to describe ...

There are two optional file-ins:
   Syntax-Color-Editor.st
      Dialog for creating own colorschemes
   Syntax-TeX-Reporter.st
      Generate _highlighted_ LaTeX-Code

                                Bert Schoenwaelder
"!


Parser subclass: #SyntaxColorParser
	instanceVariableNames: 'text inPattern commentLevel commentEnd comments '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Syntax-Highlighting'!
SyntaxColorParser comment:
'This Class does highlighting of method selector, comments and primitives while building a parse tree.

Bert Schönwälder, 1995-96
mailto:bert@cs.uni-magdeburg.de
http://www.cs.uni-magdeburg.de/~bert

Instance Variables:
	text				<Text>			the text to be colored
	inPattern		<Boolean>		true while parsing pattern
	commentLevel	<Integer>		nesting level in xDoubleQuote
	commentEnd	<nil | Integer>	end of previous comment
	comments 		<Collection of: Interval>	collected comment positions'!

!SyntaxColorParser methodsFor: 'initialize-release'!

initScanner
	super initScanner.
	inPattern := false.
	commentLevel := 0.
	commentEnd := nil.
	comments := OrderedCollection new.!

parseMethod: aText

	self text: aText.
	^self parse: (ReadStream on: text string)
		class: nil noPattern: false context: nil notifying: nil
		builder: ProgramNodeBuilder new saveComments: false ifFail: [^nil]! !

!SyntaxColorParser methodsFor: 'accessing'!

text: aText

	text := aText.! !

!SyntaxColorParser methodsFor: 'inherited'!

pattern
	"Colorize selector"

	| pattern |
	inPattern := true.
	pattern := super pattern.
	inPattern := false.
	self emphasizeFrom: 1 to: prevEnd with: #selector.
	^pattern!

readStandardPragmas: methodNode temps: temps
	"Colorize primitives and method attributes (resource, exception)"

	| start |
	start := mark.
	(super readStandardPragmas: methodNode temps: temps) isNil ifTrue: [^nil].
	self emphasizeFrom: start to: prevEnd with: #primitive.!

xDoubleQuote
	"Colorize comments"

	| val |
	"Let Parser do the work"
	commentLevel := commentLevel + 1.
	val := super xDoubleQuote.
	commentLevel := commentLevel - 1.

	"Consecutive comments are parsed recursively. Highlight all at once."
	commentLevel = 0 ifTrue: [
		| start end |
		start := prevEnd + 1.
		end := mark - 1.
		inPattern
			ifFalse: [comments add: (start to: end)]
			ifTrue: [
				commentEnd isNil ifFalse: [
					comments add: (commentEnd to: end). 
					end := commentEnd].
				self emphasizeFrom: start to: end with: #methodComment]].
	"Store start of second level commend"
	commentEnd := prevEnd.
	^val! !

!SyntaxColorParser methodsFor: 'private'!

colorizeComments

	comments do: [:i | self emphasizeFrom: i first to: i last with: #comment].!

emphasizeFrom: start to: stop with: emphSymbol

	| s last |
	s := text string.
	last := stop.
	[last < s size and: [(s at: last + 1) isSeparator]]
		whileTrue: [last := last + 1].
	text emphasizeFrom: start to: stop with: emphSymbol! !

!SyntaxColorParser class methodsFor: 'instance creation'!

parseMethod: text

	^self new parseMethod: text! !

!VariableDefinition methodsFor: 'testing'!

isInstance
	"Answer whether the receiver represents an instance variable"

	^false! !

CharacterArray subclass: #Text
	instanceVariableNames: 'string runs '
	classVariableNames: 'ColoringAttributes '
	poolDictionaries: 'TextConstants '
	category: 'Collections-Text'!
Text comment:
'Class Text handles protocol for treating strings of characters as displayable characters that can have emphasis and font changes.  The emphasis codes indicate abstract changes in character appearance.  Actual display is performed in the presence of a TextAttributes which indicates, for each abstract code, an actual font to be used.

Instance Variables:
	string	<String> of Characters
	runs	<RunArray> of emphasis codes

Pool Dictionaries:
	TextConstants	<Dictionary> of associations betwen key names and characters'!

!Text methodsFor: 'emphasis'!

makeSelectorBoldIn: aClass
	"For formatting Smalltalk source code, set the emphasis of that portion of 
	the receiver's string that parses as a message selector to be bold."
	"Changed: use <colorMethodFor:> if ctrl isn't pressed"

	| parser |
	string size = 0 ifTrue: [^self].
	InputState default ctrlDown
		ifFalse: [ ^self coloredMethodFor: aClass].
	(parser := aClass parserClass new) parseSelector: string.
	self emphasizeFrom: 1
		to: (parser endOfLastToken min: string size)
		with: #bold! !

!Text methodsFor: 'coloring-methods'!

coloredMethodFor: aClass
	"This method uses the Smalltalk parsing capabilities for (CPU dependent) fast, 
	(hopefully) correct and (of course) detailed syntax higlighting."

	"Change class method initColoringAttributes for other Attributes, or (if you got it) do:"
	"SyntaxColorEditor open"

	self symbolicColorsFor: aClass.
	runs translateValues: [:v | v == nil ifFalse: [Text coloringAttributes at: v ifAbsent: nil]]!

simpleSymbolicColorsFor: aClass
	"Highlight a method when symbolicColors failed"

	| parser methodComment endOfSelector |
	parser := aClass parserClass new.
	Object errorSignal
		handle: [:ex |
			parser parseSelector: string.
			endOfSelector := parser endOfLastToken min: string size]
		do: [ 
			"first all comments"
			parser scanComments: string do: [:start :stop | 
				self emphasizeFrom: start to: stop with: #comment].
			"parse pattern"
			methodComment := parser parseMethodComment: string setPattern: [:x|].
			endOfSelector := parser endOfLastToken min: string size.
			"method comment"
			methodComment isEmpty ifFalse: [
				self emphasizeFrom: endOfSelector
					to: endOfSelector + methodComment first size + 2		"the 2 double quotes"
					with: #methodComment]].
	"selector"
	self emphasizeFrom: 1 to: endOfSelector with: #selector.!

symbolicColorsFor: aClass
	"Creates runs of symbols according to token types"

	| parser node |
	Object errorSignal
		handle: [:ex | self simpleSymbolicColorsFor: aClass]
		do: [
 			"colorize selector, methodComment and primitive, build parse tree, collect comment intervals"
			parser := SyntaxColorParser new.
			node := parser parseMethod: self.
			"colorize variables and literals"
			node notNil ifTrue: [
				SyntaxColorEnumerator enumerate: node text: self class: aClass].
			"colorize comments"
			parser colorizeComments].! !

!Text class methodsFor: 'coloring-attributes'!

coloringAttributes
	ColoringAttributes isNil ifTrue: [self initColoringAttributes].
 	^ColoringAttributes!

coloringAttributes: someColoringAttributes
	ColoringAttributes := someColoringAttributes!

coloringAttributesOrganization
	"Answer a Organizer-like array for use in SyntaxColorEditor"

	^#(
	(Variable (argument temporary instance class global undeclared))
	(Literal (number array string symbol character constant))
	(Misc (comment methodComment selector primitive)))!

initColoringAttributes
	"Initialize attributes for syntax highlighting"
	"Text initColoringAttributes"

	"If you dont want a special thing to be highlighted, just put nil instead of the array into it.
	A text is displayed slower if it has many small runs of attributes, so if this a problem 
	for your CPU you should consider niling the literals or globals, for example.
	After changing, don't forget to execute the comment above!!"

	(ColoringAttributes := Dictionary new)
		"Method pattern"
		at: #selector put: (Array with: #bold);
		at: #primitive put: (Array with: #color->ColorValue blue);
		"Variables in scope ..."
		at: #temporary put: (Array with: #color -> (ColorValue hue: 0.5 saturation: 1 brightness: 0.5));
		at: #argument put: (Array with: #color -> (ColorValue hue: 0.4 saturation: 1 brightness: 0.5));
		at: #instance put: (Array with: #color -> (ColorValue hue: 0 saturation: 1 brightness: 0.6));
		at: #class put: (Array with: #color -> (ColorValue hue: 0 saturation: 1 brightness: 0.6));
		at: #global put: (Array with: #color -> (ColorValue hue: 0.7 saturation: 1 brightness: 0.6));
		at: #undeclared put: (Array with: #color -> ColorValue red with: #bold);
		"Literals: constants (nil, true, false), symbols, strings, characters, arrays, numbers"
		at: #constant put: (Array with: #color -> (ColorValue hue: 0.7 saturation: 1 brightness: 0.8));
		at: #symbol put: (Array with: #color -> (ColorValue hue: 0.1 saturation: 1 brightness: 0.7));
		at: #string put: (Array with: #color -> (ColorValue hue: 0.1 saturation: 1 brightness: 0.7));
		at: #character put: (Array with: #color -> (ColorValue hue: 0.1 saturation: 1 brightness: 0.7));
		at: #array put: (Array with: #color -> (ColorValue hue: 0.1 saturation: 1 brightness: 0.7));
		at: #number put: (Array with: #color -> (ColorValue hue: 0.1 saturation: 1 brightness: 0.7));
		"Comments"
		at: #comment put: (Array with: #italic with: #color -> ColorValue darkMagenta);
		at: #methodComment put: (Array with: #italic with: #color -> ColorValue darkMagenta with: #bold).! !

!InstanceVariable methodsFor: 'testing'!

isInstance
	"Answer whether the receiver represents an instance variable"

	^true! !

ProgramNodeEnumerator subclass: #SyntaxColorEnumerator
	instanceVariableNames: 'text class method variables '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Syntax-Highlighting'!
SyntaxColorEnumerator comment:
'This class is used in "Text>symbolicColorsFor:" to do syntax highlighting for variables and literals. See doLiteral... and doVariable... in <enumerating>. 

Bert Schönwälder, 1995-96
mailto:bert@cs.uni-magdeburg.de
http://www.cs.uni-magdeburg.de/~bert

Instance Variables:
	text			<Text>	the text to be colorized
	class		<Class>	the class for which highlighting is done
	method		<CompiledMethod>	the compiled method in class
	variables	<Dictionary>	classification of variables
'!

!SyntaxColorEnumerator methodsFor: 'initialize-release'!

colorize: aProgramNode text: aText class: aClass
	"Set up and run"

	text := aText.
	class := aClass.
	variables := Dictionary 
		with: 'self' -> #instance
		with: 'super' -> #instance
		with: 'thisContext' -> #instance.
	self doNode: aProgramNode.! !

!SyntaxColorEnumerator methodsFor: 'enumerating'!

doAssignment: aNode variable: var value: val

	self doNode: val.
	self doNode: var.!

doBlock: aNode arguments: args body: seq

	args do: [:var | variables at: var name put: #argument].
	self doNodes: args.
	self doNode: seq.!

doCascade: aNode receiver: rcvr messages: msgs

	self doNode: rcvr.
	self doNodes: msgs!

doLiteral: aNode value: lit

	"Colorize literals"
	"Classify them"
	self emphasize: aNode sourcePosition with: (
		lit isSymbol ifTrue: [#symbol] 
			ifFalse: [lit isString ifTrue: [#string] 
				ifFalse: [(lit isKindOf: Number) ifTrue: [#number] 
					ifFalse: [(lit isKindOf: Character) ifTrue: [#character] 
						ifFalse: [(lit isKindOf: ArrayedCollection) ifTrue: [#array] 
							ifFalse: [#constant]]]]])!

doMessage: aNode receiver: rcvr selector: sel arguments: args

	self doNode: rcvr.
	self doNodes: args!

doMethod: aNode selector: sel primitive: prim block: block

	method := class compiledMethodAt: sel ifAbsent: [].
	prim notNil ifTrue: [
		| err |
		(err := aNode primitiveErrorCode) notNil ifTrue: [
			variables at: err name put: #temporary.
			self doNode: err]].
	self doNode: block!

doParameter: aNode variable: var type: type

	self doNode: var!

doReturn: aNode value: value

	self doNode: value!

doSequence: aNode temporaries: temps statements: stats

	temps do: [:var | variables at: var name put: #temporary].
	self doNodes: temps.
	self doNodes: stats.!

doVariable: aNode name: nameString
	"Colorize variables"
	"Temporaries and arguments are classified yet (see doSequence... and doBlock...), so
	we need to check for global, instance, and class variables only"

	self emphasize: aNode sourcePosition with:
		 (variables at: nameString ifAbsentPut: 
			[self scopeFor: nameString node: aNode])! !

!SyntaxColorEnumerator methodsFor: 'private'!

bindingFor: variableName
	"Answer the binding for var in the compiled method"

	| variableSymbol |
	method isNil ifTrue: [^nil].
	variableSymbol := variableName asSymbol.
	method withAllBlockMethodsDo: [:m |
		m literalsDo: [:binding |
			(binding isVariableBinding and: [variableSymbol == binding key]) ifTrue: [^binding]]].
	^nil!

emphasize: anInterval with: anAttributeSymbol

	| s last |
	s := text string.
	last := anInterval last.
	[last < s size and: [(s at: last + 1) isSeparator]]
		whileTrue: [last := last + 1].
	text emphasizeFrom: anInterval first  to: last with: anAttributeSymbol!

scopeFor: variableName node: aNode
	"We know variable is neither temporary, argument nor pseudo"

	| varDef binding |
	"Make NameScope. It's cached, fortunately"
	varDef := (NameScope forClass: class) variableAt: variableName from: aNode.
	varDef isNil ifTrue: [^#undeclared].
	"It's either an instance ..."
	varDef isInstance ifTrue: [^#instance].
	"... or a static binding"
	varDef isStatic ifFalse: [^#undeclared "should not happen"].
	"Check if the binding in the compiled method is the same as in the scope"
	binding := self bindingFor: variableName asSymbol.
	(binding isNil or: [binding ~~ varDef binding]) ifTrue: [^#undeclared].
	"Binding is okay, check for global"
	(Smalltalk includesKey: variableName) ifTrue: [^#global].
	"It's in some pool. Don't bother in whitch ..."
	^#class! !

!SyntaxColorEnumerator class methodsFor: 'class initialization'!

initialize
	"Disable the Application Management's comment highlighting, if present"
	"If there are other packages to include here, please mail me:
		bert@cs.uni-magdeburg.de"

	| appBrowser |
	appBrowser := (Smalltalk at: 'ApplicationBrowser' ifAbsent: []).
	(appBrowser notNil and: [appBrowser respondsTo: #userPreferencesAt:put:]) ifTrue: [
		appBrowser userPreferencesAt: #commentEmphasis put: nil].! !

!SyntaxColorEnumerator class methodsFor: 'instance creation'!

enumerate: aProgramNode text: aText class: aClass
	"Simply run it"

	self new 
		colorize: aProgramNode
		text: aText
		class: aClass! !

!MethodNode methodsFor: 'accessing'!

primitiveErrorCode

	^primitiveErrorCode! !

!RunArray methodsFor: 'text support'!

translateValues: translateBlock
	"Replace values with results from translateBlock"

		values keysAndValuesDo: [:i :v |	values at: i put: (translateBlock value: v)].! !

SyntaxColorEnumerator initialize!


